#include <avr/sleep.h> // power-down
#include <util/delay_basic.h> // _delay_loop_2(t) ritardo (1/clock)*4*t
#include <avr/eeprom.h>
 
const uint8_t buttons[4] = { // PORTB: accende il LED
  0b00001010, 0b00000110, 0b00000011, 0b00010010
};
const uint8_t tones[4] = { // frequenza PWM per le note del buzzer
  239, 179, 143, 119
};
uint8_t lastKey; // debouncing pulsanti
uint8_t lvl = 0; // livello raggiunto
uint8_t maxLvl;  // livello massimo (high-score su EEPROM)
uint16_t seed; // seed generatore numeri pseudo-casuali
uint16_t ctx;  // valori pseudo-casuali generati col seed
volatile uint8_t nrot = 8; // quante volte si mischia il seed
volatile uint16_t time;    // incrementata ogni 16ms (interrupt WDT)
 
void sleepNow() {     // power-down
  PORTB = 0b00000000; // tri-state
  cli();              // disabilita gli interrupt
  WDTCR = 0;          // spegne il WDT
  set_sleep_mode(SLEEP_MODE_PWR_DOWN);
  sleep_enable();
  sleep_cpu();
}
 
void play(uint8_t i, uint16_t t = 45000) {
  PORTB = 0b00000000; // tri-state (input) e livello L (output)
  DDRB = buttons[i];  // pin del buzzer e di un LED come uscita
  OCR0A = tones[i];      // imposta frequenza PWM
  OCR0B = tones[i] >> 1; // duty-cycle 50%
  TCCR0B = (1 << WGM02) | (1 << CS01); // PWM phase-correct prescaler 8
  _delay_loop_2(t); // LED acceso e nota suonata per 150ms (o in base a t)
  TCCR0B = 0b00000000; // PWM off
   DDRB = 0b00000000; // LED e buzzer off (tutti ingressi)
  PORTB = 0b00011101; // pull-up sui pulsanti
}
 
void gameOver() {
  for (uint8_t i = 0; i < 4; i++) {
    play(3 - i, 25000); // accende i 4 LED all'indietro
  }
  if (lvl > maxLvl) { // scrive su EEPROM nuovo record (complementato)
    eeprom_write_byte((uint8_t*) 0, ~lvl); 
    for (uint8_t i = 0; i < 3; i++) { // animazione high-score
      levelUp();
    }
  }
  sleepNow();
}
 
void levelUp() {
  for (uint8_t i = 0; i < 4; i++) { // accende in sequenza i 4 LED
    play(i, 25000);
  }
}
 
uint8_t simple_random4() { // LFSR (Galois) pseudo-random generator 
  for (uint8_t i = 0; i < 2; i++) { // aggiorna ctx
    uint8_t lsb = ctx & 1; 
    ctx >>= 1;
    if (lsb || !ctx) {
      ctx ^= 0xB400;
    }
  }
  return ctx & 0b00000011; // restituisce un numero tra 0 e 3
}
 
ISR(WDT_vect) { // Interrupt Server Routine eseguita ogni 16ms
  time++; // per debounce e power-down
  if (nrot) { // mischia 8 volte il seed col valore del contatore
    nrot--;
    seed = (seed << 1) ^ TCNT0;
  }
}
 
void resetCtx() {
  ctx = seed; // riporta ctx al valore iniziale, quello del seed
}
 
int main(void) {
  PORTB = 0b00011101; // pull-up sui 4 pulsanti
  ADCSRA |= (1 << ADEN); // abilita l'ADC
  ADCSRA |= (1 << ADSC); // parte la conversione sul pin ADC0 scollegato
  while (ADCSRA & (1 << ADSC)); // attende fine conversione
  seed = ADCL; // 8 bit meno significativi del valore ottenuto
  ADCSRA = 0b00000000; // spegne l'ADC
  WDTCR = (1 << WDTIE); // attiva watchdog timer con prescaler da 16ms
  sei(); // interrupt enable
  TCCR0B = (1 << CS00); // timer/counter in normal mode a 1,2MHz
  while (nrot); // mischia il seed usando il valore del counter
  TCCR0A = (1 << COM0B1) | (0 << COM0B0) | (0 << WGM01)  | (1 << WGM00); 
  maxLvl = ~eeprom_read_byte((uint8_t*) 0); // lettura high-score
  switch (PINB & 0b00011101) {
    case 0b00010101: // pulsante LED rosso dopo l'avvio
      eeprom_write_byte((uint8_t*) 0, 255); // reset high-score
      maxLvl = 0;
      break;
  }
 
  while (1) {
    resetCtx(); // parte da seed
    for (uint8_t cnt = 0; cnt <= lvl; cnt++) { // fino a lvl
      _delay_loop_2(4400 + 489088 / (8 + lvl)); // sempre più veloce
      play(simple_random4()); // aggiorna ctx e accende un LED 
    }
    time = 0; // tempo debounce
    lastKey = 5; // valore che non corrisponde a nessun pulsante
    resetCtx(); // riparto da seed
    for (uint8_t cnt = 0; cnt <= lvl; cnt++) { // fino a lvl
      bool pressed = false; // azzero pressed
      while (!pressed) {    // finché non si preme un bottone
        for (uint8_t i = 0; i < 4; i++) { // provo i quattro bottoni
          // PINB ha tutti i bit a 1 tranne quello del pulsante premuto
          // buttons[i] & 0b00011101 tutti zero tranne il pulsante i
          if (!(PINB & buttons[i] & 0b00011101)) {
            // DEBOUNCE
            if (time > 1 || i != lastKey) { // 16ms o nuovo pulsante
              play(i); // LED + suono
              pressed = true; // registra pressione
              uint8_t correct = simple_random4(); // ri-genera l'indice
              if (i != correct) { // premuto il pulsante sbagliato
                for (uint8_t j = 0; j < lvl; j++) { // per lvl volte
                  _delay_loop_2(65536);
                  play(correct, 45000); // accende il LED giusto
                }
                _delay_loop_2(65536);
                gameOver();
              } // altrimenti è stato premuto il pulsante giusto 
              time = 0; // azzera time
              lastKey = i; // memorizzo l'indice dell'ultimo LED
              break; // esce dal ciclo for e passa al LED successivo 
            }
            time = 0; // azzera il conteggio per poweroff e debouncing
          }
        }
        if (time > 4000) { // dopo 64s senza pressione di un tasto
          sleepNow(); // power-off
        }
      }
    }
    _delay_loop_2(65536);
    if (lvl < 254) { // inserita la sequenza corretta
      lvl++; // aumenta il livello
      levelUp(); // animazione levello completato
      _delay_loop_2(45000);
    }
  }
}
